function Surtsey_One_k_erfIC_forArXiv
%% Uses the method of lines to solve nondim model of Surtseyan Bomb Fragmentation
% Emma Greenbank and Mark McGuinness
%
% A coordinate tranformation is used to transform a variable mesh to a uniform
% mesh. In the original (frozen) coordinates, the mesh is finer next to the
% flash point, where the most significant gradients are seen.
% This version solves for one permeability and porosity value, generating
% plots like those seen in Fig 6 of the arXiv manuscript.
% 
% A ramped initial temperature based on the erf similarity solution is used, 
% but a stepped initial pressure is used, set at flash to match the temperature there,
% but at atmospheric pressure elsewhere in the magma, so that density will
% be set by the values of P, T in that region as it is elsewhere in the magma.
% The default is to set the temperature RampDistance to 1E-05/R2, that is, ten
% microns, being a typical pore size for pumice. Also to set the smallest mesh
% size to be about 1/10 of this for numerical resolution. 
% Best to use at least 800 mesh points in magma for accuracy.
%
% This version uses just one k and phi value, and is intended to accompany an
% arXived version of our manuscript that is accepted for publication in
% Proc Roy Soc London A (August 2021), entitled 
% "A theoretical model of Surtseyan bomb fragmentation"
%

global T0 R2 R1 s0 p2 Tnp1 Tn Tm2 Tm3 delta5 epsilon3 RampDistance pmaxtheor tmaxtheor

m = 400; % the number of mesh partitions in the magma (800 recommended)
n=m/2; % slurry mesh number

%options=odeset('RelTol',1.0e-8,'AbsTol',1.0e-8,'events',@MyEvents);
% MyEvents detects a local maximum in pressure adjacent to flash point, and
% stops iteration upon detection
options=odeset('RelTol',1.0e-8,'AbsTol',1.0e-8); % don't stop at max pressure of flash point

k = 1.0E-12; 
phi=0.3;

tmax=0.01; % stop running at tmax, dimensionless time. It is 1 when all slurry is evaporated.
nft=51; finet = zeros(1,nft);

% use a transform to refine mesh near flash point. Use an evenly (unit)
% spaced variable chi for slurry ranging from 0 to n+1 in value, so there
% are n+2 indices in total, n ode variables plus two boundary values. 
% Use psi for hot magma ranging from 0 to m+1 in value, unit spaced like
% chi, with m variables solved for in odes, and two boundary values.

RescaleSurtsey;  % generates epsilon and delta values, and pmaxtheor, tmaxtheor. 
                 % RampDistance is set in this call to RescaleSurtsey

smallestMesh=RampDistance/10; % the smallest mesh size to be used

tSince = RampDistance^2/delta5/16; % time since emplacement and step function initial conditions

ximin=smallestMesh/(1-s0);  % s0=R1/R2 is the initial dimless value of s
zetamin = smallestMesh/s0;    % the smallest mesh in zeta                       
                        
% now set aam, given m, so that the smallest mesh size in the magma is roughly smallestMesh:
% set up the mesh size in the magma, using the specified m. 
         % find an approximate value for aam that ensures the mesh is not
         % too small near the flash point:
% 
aam = 2/(pi*(m+1)*(m+1)*ximin);  % this is only approximate, and only in magma

myGetAAmag = @GetZmag; %the zero of this function gives an accurate "a" value
aam = fzero(myGetAAmag,[aam/2 aam*2]); % an accurate value for aam, in magma

% get a different aam value for slurry, having first chosen a value for n
aas=2/(pi*(n+1)*(n+1) * zetamin); %this is only a rough estimate, to help

myGetAAslurry = @GetAAslurry; %the zero of this function gives an accurate "a" value
aas = fzero(myGetAAslurry,[aas/2 aas*2]); % an accurate value for aam in slurry
                        
Bm = atan(aam*(m+1)); Bn = atan(aas*(n+1)); %constants needed later

chi=linspace(0,n+1,n+2); % 0,1,2,3, ... n+1 in value
psi=linspace(0,m+1,m+2); % 0,1,2,3, ... m+1 in value
zeta=atan(aas*chi)/atan(aas*(n+1));  % zeta in slurry goes from 0 to 1, tightens on 1
xi= 1 + atan(aam*(psi-m-1))/atan(aam*(m+1)); % xi in hot magma goes from 0 to 1, 
                                           % and tightens mesh on 0
Cm = Bm/aam*(1+aam*aam*(psi-m-1).*(psi-m-1)); % vector term that arises from chain rule

Cn = Bn/aas*(1+aas*aas* chi.*chi); % vector term that arises from chain rule                                          

if (xi(2)-xi(1)) < smallestMesh/2   % possibly n is too large, smallest spacing is too small
    disp('bailing out, smallest mesh size is too small, maybe n is too large?')
    disp(xi(1:3))
    display(StopNow)
end

y0=InitialConds(n,m);  % set up the initial conditions

    s = y0(n+2*m+1);  % flash location is fed into FlashBC as global
    Tm2 = y0(n+1); % leftmost temperature in magma, needed by FlashBC, global
    Tm3 = y0(n+2);  % next temperature, also needed for 2nd order first difference approxn to gradient
    p2= y0(n+m+1)*Tm2 ;   % needed by FlashBC, passed as global
    p3=y0(n+m+2)*Tm3; % also needed by FlashBC
    Tnp1 = y0(n); % temperature in slurry next to flash, needed by FlashBC, global
    Tn= y0(n-1);  % temperature in slurry, two nodes from flash, needed by FlashBC, global

%======================================================================
% solve the problem, using one value for k and phi:

soln=ode15s(@SurtseyModel,[0 tmax], y0, options);

tmaxt = 0.9999*soln.x(end);  % the last time value returned by ode15s
tmin = tmaxt/1E08;
finet(2:end) = logspace(log10(tmin),log10(tmaxt),nft-1); %finer times, log spaced, for s, pflash, tflash vs time

FineVals = deval(soln,finet); % get solution at times finet


% note that deval returns each time in a different column.
% Unpacksolutions fails if no of rows = no of columns AND different times
% are in different columns. It wants different times on diff rows.
% so ensure that FineVals and CoarseVals have diff times on diff rows: 
[Tincl,Tmag,rhomag,ess] = UnpackSolutions(FineVals'); %gets T, rho, s from solutions
                                                    
pmag = rhomag.*Tmag; % pressure in hot magma region

% watch out for premature finish:

if (tmax- soln.x(end)) < 1E-05*tmax
    disp('warning Will Robinson:')
    disp('did not run long enough')
    disp('max run time is ')
    disp(tmax)
end

% plot results for these k, phi values:

figno=1; defFont=18;

RNslurry= ess.*zeta; % the nondimensional r values in the slurry, moving s
RNmagma=(ess+(1-ess).*(xi));  % in the magma. sc is a row vector depending on time; 
                            % xi is a column vector;
                            % hence each row of the resulting matrix RN is
                            % a different time


figure(figno) % plot temperatures in slurry inclusion as mesh
clf('reset')
colormap jet;
mesh(RNslurry,log10(finet),Tincl,'LineWidth', 2)
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
zlabel('T');xlabel('r'); ylabel('log10 t');
title('T in slurry')
view(230,30)

figno=figno+1;
figure(figno) %plot temperatures in hot magma, as a mesh plot
clf('reset')
colormap jet;
mesh(RNmagma,log10(finet),Tmag,'LineWidth', 2) % multiple lines, one for each time, will use diff colours for each?
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
zlabel('T');xlabel('r'); ylabel('log10 t');
title('T in magma')
view(-30,30)

figno=figno+1;
figure(figno) % mesh plot of pressures in hot magma
clf('reset')
colormap jet;
mesh(RNmagma,log10(finet),pmag,'LineWidth', 2) % multiple lines, one for each time, will use diff colours for each?
ax=gca;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times'; 
zlabel('P');xlabel('r'); ylabel('log10 t');
title('P in magma')
view(60,30)

figno=figno+1;
figure(figno) % let's have a look at the location of the flashing front
clf('reset')
plot(finet,ess,'-k','LineWidth', 3) % one line for s(t)
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('s');xlabel('t');

figno=figno+1;
figure(figno) % the pressure at the flashing front, vs time:
clf('reset')
plot(finet,pmag(:,1),'-k','LineWidth', 3) % one line for p at flash
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('p flash');xlabel('t');

figno=figno+1;
figure(figno)  % plot the temperature at the flash point too
clf('reset')
plot(finet,Tmag(:,1),'-k','LineWidth', 3) % one line for T at flash
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('T flash');xlabel('t');

figno=figno+1; 
% plot temperatures in magma vs similarity variable. Compare to theoretical erfc solution
% note that there is a range of temperatures at the flash point to cope
% with, and that late times give poor matches to similarity solution,
% mainly because the flash boundary condition goes up and down in reality,
% but the similarity solution assumes a step change in boundary temperature

figure(figno)  
clf('reset')

tvalsHere=finet';
timend=floor(length(finet)/1); % dividing by a number bigger than one allows
                               % cut out very late times
sigstart=floor(m/5); % cut out locations very close to the flash front
sigma=(1-ess).* xi /sqrt(delta5); % has one row per time, and a different 
                                 % column for each spatial coord value
eta = (sigma(2:timend, sigstart:end).^2) ./(tvalsHere(2:timend)); % eta is similarity variable

plot(eta,Tmag(2:timend,sigstart:end),'or','MarkerSize',10,'LineWidth',2 ) % simulated temperatures vs eta

% also plot the theoretical similarity solution for max and min flash
% temperatures:

Tffmin=Tmag(2,1);  %the smallest flash temperature
etamin=0.001; etamax=50.0;
etath=linspace(etamin,etamax,500);

theoryTm=(1-Tffmin)*erf(sqrt(etath)/2.0)+ Tffmin;
hold on;
plot(etath,theoryTm, '-k','LineWidth',4)

Tffmax=max(Tmag(:,1)); % the maximum flash temperature seen 
theoryTm=(1-Tffmax)*erf(sqrt(etath)/2.0)+ Tffmax;
plot(etath,theoryTm, '-k','LineWidth',4)

hold off;
ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('T magma');xlabel('\eta');
xlim([etamin etamax]);

% now plot temperatures vs eta in the slurry inclusion:
sigma2=ess.*(1-zeta)/sqrt(epsilon3); % ess is a col vector, s(t); zeta is a row vector;
                                       % this product has one row for each
                                       % time, it is implicitly expanded
figno=figno+1; 
figure(figno)
clf('reset')
timend=floor(length(finet)/1.5); % don't use later times
sigstart=floor(m/5);
eta = (sigma2(2:timend, sigstart:end).^2)./(tvalsHere(2:timend)); % this works OK because tvals is a row vector and
                              % Matlab sees that tvals' has one row and
                              % sigma2 has the same number of rows (columns
                              % are mesh/spatial);
                              % So Matlab makes the result have length(tvals) rows
                              % and length(sigma2) columns, intuitively

plot(eta,Tincl(2:timend,sigstart:end),'or','MarkerSize',10,'LineWidth',2 ) % multiple lines, one for each time


theoryTinc=(T0-Tffmin)*erf(sqrt(etath)/2.0)+ Tffmin;
hold on;
plot(etath,theoryTinc, '-k','LineWidth',4)

Tffmax=max(Tmag(:,1)); % bracket the range of flash temperatures seen 
theoryTinc=(T0-Tffmax)*erf(sqrt(etath)/2.0)+ Tffmax;
plot(etath,theoryTinc, '-k','LineWidth',4)

hold off;

ax=gca;
ax.FontSize=defFont; ax.FontName='Times'; 
ylabel('T slurry');xlabel('\eta');
xlim([etamin etamax]);

    function y0=InitialConds(n,m)
        %% InitialConds is to set up the initial values of the variables

        global p0 Haitch
        
        p0 = 1; %initial pressure
%        Ti=1; % initial temperature in magma if no ramping
%         Tslurry=T0; % inital temperature in slurry if no ramping
%         
%         rhoz=p0/Ti; % initial density of vapour in magma
%         
%         y0=zeros(n+2*m+1,1); % set up variables as column vector
%         
%         y0(1:n) = Tslurry; % the first n are T in slurry
%         y0(n+1:n+m) = Ti;       % the next m are T in magma
%         y0(n+m+1:n+2*m) = rhoz; % the next m are density in magma


        % but rather than a step change across the flash point, I want to 
        % ramp up rapidly from Tslurry to Ti, in the magma, and down from Ts
        % to T0 in the slurry as we move away from the flash point. 
        % The similarity solutions give values over the full distances, but they 
        % effectively ramp over the distance RampDistance in each medium. This
        % distance is the (nondimensional) typical pore size, of the order of the size of a
        % representative elementary volume over which the rock matrix and
        % the fluid in the pores are averaged. RampTimes and RampTimem are 
        % the equivalent times from step form of the error functions, in
        % slurry and in magma respectively.
        % I  need the Tflash value; call UnpackSolutionsInitial for this, 
        % which is based on UnpackSolutions but uses the error function form
        % to compute gradients at flash.
        % then replace all T values with ramping values.
        % I won't ramp the pressures in the magma; the densities will
        % accommodate the resulting pressures and temperatures. Note that
        % the pressure gradient affects the just-calculated Tflash
                           
            p2= p0 ;   % needed by FlashBCinit, passed as global
            p3= p0 ;   % needed by FlashBCinit, passed as global
            
            % check straddle zero. Look between slurry temp and critical
            % temp

            r1=FlashBCinit(0.25);
            r2=FlashBCinit(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero?? inside initial conds 1')
                disp(StopNow)
            else % we do have straddle: find initial flash temperature
                Tflash = fzero(@FlashBCinit,[0.25 0.5]); % solve BC at flash for temperature there
            end
            
%             Lookie = LookAtFlashBCInit(Tflash);
%             format shortEng
%             disp('result of look at Flash BC init:')
%             disp(Lookie)
%        
        % Tflash depends on the temperature gradients on either side, and on the P gradient 
        % in the magma. These
        % are known - I have specified the distance over which initial T ramps
        % up, and I know what it ramps to. So Tflash should line up OK.
        
        RNslurr= s0*zeta(2:n+1); % the mesh in the slurry at time zero; nondimensional r values
                                 % the first and last zeta(1:n+2) is at boundary
                                 % values, not y0 values
        RNmag=s0+(1-s0)*(xi(2:m+1));  % the mesh in the magma at time zero, in terms of the 
                         % nondimensional R values, normalised on R2. First
                         % and last values are boundary locations, not y0
 
% start with similarity solution for T in magma and in slurry:
        Tmagma = Tflash + (1-Tflash)*erf((RNmag-s0)/(2*sqrt(delta5*tSince)));
        
        Tslurry = Tflash + (T0-Tflash)*erf( (s0-RNslurr)/(2* sqrt(epsilon3*tSince)));
        
        y0(1:n) = Tslurry; % the first is set by the BC at origin, the last is Tflash
        y0(n+1:n+m) = Tmagma; % the first is Tflash, the last is the boundary value, close to 1
        y0(n+m+1:n+2*m) = p0./Tmagma;        % densities in magma
        y0(n+2*m+1) = s0;  % initial position of flash front s
        
         Pflash = exp(Haitch*(Tflash - T0)./Tflash);
         disp('initial conditions, Pf and Tf, using erf slopes:')
         disp(Pflash)
         disp(Tflash)
         
         % check what the flash conditions will be with these surrounding
         % p, T values:
         Tnp1 = Tslurry(n);
         Tn = Tslurry(n-1);
         Tm2 = Tmagma(1);
         Tm3 = Tmagma(2);
         s=s0;
         
            r1=FlashBC(0.25);
            r2=FlashBC(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero?? inside initial conds 2')
                disp(StopNow)
            else % we do have straddle: find initial flash temperature
                Tflash2 = fzero(@FlashBC,[0.25 0.5]); % solve BC at flash for temperature there
            end
          Pflash2 = exp(Haitch*(Tflash2 - T0)./Tflash2);
          disp('recalculated flash conditions using initial conditions, Pf2 and Tf2:')
          disp(Pflash2)
          disp(Tflash2)        
       
    end

    function [Ts,Tm,rho,ess] = UnpackSolutions(y)
        %% UNPACKSOLUTIONS takes the y values solved for, and unpacks them
        % y unpacks to T values in slurry, T values in magma, rho values in 
        % magma, and flashing front location s(t)
        % values at boundaries are also added
        % I need to ensure this can handle a single vector y, and a matrix
        % with rows or columns of values, one for each of a series of times
        
        global  Haitch  p0 Tsurface
        
        ysize=size(y);  % many rows, 1 column usually if solving. 
 
        nrows = ysize(1);
        ncols = ysize(2); % handles multiple rows/times ok too
        
        % make sure different rows are different times, so the number of columns
        % is n+2m+1, by finding whether
        % rows or columns match the expected length n+2m+1:
        if (nrows == n+2*m+1) && (ncols ~= n+2*m+1)  %uh oh, each column is a time
            y=y';    % switch rows and columns; don't switch if by some chance, 
                     % both the number of times
                     % and the number of variables match up; in that case it is up
                     % to the programmer to ensure different rows are at
                     % different times.

            nrows=ncols; % because you just switched rows and cols. Only need
                         % nrows now
 
        end
        
        % now each row is a different time. There may be only one row:
        
        Ts(:,2:n+1)= y(:,1:n); % temperatures in the slurry, excluding the boundaries
        Tm(:,2:m+1) = y(:,n+1:n+m); % temperatures in the magma, excl boundaries
        rho(:,2:m+1) = y(:,n+m+1:n+2*m); % vapour density in magma
   
        
        % now add the boundary values on to the solved values
        
        pv2 = rho(:,2).*Tm(:,2);  % pressures in magma adjacent to flash, needed for FlashBC
        pv3 = rho(:,3).*Tm(:,3);  % pressures in magma 2 nodes from flash, needed for FlashBC
        Ts(:,1) = Ts(:,2); % no heat flow at origin, in slurry
        ess = y(:,n+2*m+1);    % flash locations
        
        % use a for loop to find the flash temperatures:
        Tflash = zeros(nrows,1);
        for i=1:nrows   %each row is a different time; there may be only one 

            s = ess(i);                % flash location fed into FlashBC as global
            Tm2 = Tm(i,2); % leftmost temperature in magma, needed by FlashBC, global
            Tm3 = Tm(i,3);
            p2= pv2(i) ;   % needed by FlashBC, passed as global
            p3= pv3(i) ;   % needed by FlashBC, passed as global
            Tnp1 = Ts(i,n+1); % temperature in slurry next to flash, needed by FlashBC, global
            Tn = Ts(i, n); 
            % check straddle zero. Look between slurry temp and critical
            % temp
            %r1=FlashBC(T0);
            r1=FlashBC(0.25);
            r2=FlashBC(0.5); %T=0.5 is approximately critical temperature
            if r1*r2 > 0   % same sign, not straddling!!
                disp('not straddling zero; s,Tm2,p2,Tn+1:')
                disp(s)
                disp(Tm2)
                disp(p2)
                disp(Tnp1)

                Tflash(i) = fzero(@FlashBC,Tnp1); % look for a solution near the slurry T
                disp(Tflash(i))
                LookAtFlash
                disp(StopNow)
            else % we do have straddle:
            Tflash(i) = fzero(@FlashBC,[0.25 0.5]); % solve BC at flash for temperature there
            end
  
        end  
        
        Ts(:,n+2) = Tflash; Tm(:,1) = Tflash; % temperatures match at flash point
        
        % other values can now be computed at the flash point:
        Pflash = exp(Haitch*(Tflash - T0)./Tflash);
        % 
        rho(:,1) = Pflash./Tflash; 
        
        % values at surface of bomb:
         
        Tm(:,m+2) = Tsurface;  % set T of vapour at magma surface (to boiling point at one atm)  
        rho(:,m+2) = p0/Tsurface; % boundary value of vapour density  
        
    end 

    function RSide = SurtseyModel(~, y)
        %% SurtseyModel sets up the RHS of the differential equations
        %  RSide should be a column vector of same length as initial
        %  conditions. It doesn't depend on time, hence the ~.
        
        
        global epsilon4 epsilon5 
                                 
        % here are the actual variables in the ODEs:
        % unpack y to T, Tm, rho, s values (T slurry-n terms in y, T magma-m terms in y, density of
        % vapour in magma-m terms, location of flashing front-1 term). Add
        % two boundary values to Temperatures and density
        
        [T,Tm,rho,s] = UnpackSolutions(y); % This generates boundary values too
                                           % so LHS winds up with more
                                           % terms than y has      
        p = rho.*Tm;  % pressures in magma, the full set from 1 to m+2
%        
% the above pressure is correct at the flash point, where pressure is set
% by the clausius clapeyron formula. It is OK since I have set the value of
% rho here using a pressure from Clausius Clapeyron, anyway.

        %dsdt = epsilon4*Cm(1)*rho(1)/(1.0-s) * (p(2)-p(1)); 
        
        % use a second-order accurate one-sided difference for dp/d chi:
        dsdt = epsilon4*Cm(1)*rho(1)/(1.0-s) * (-3*p(1)+4*p(2)-p(3))/2;                                      
          
        % Now compute the RHsides of the differential equations:
        %
        % T in slurry: there are n terms plus 2 boundary values in the
        % slurry
        %
        
        vT = -(zeta(2:n+1).* zeta(2:n+1)*s*dsdt + 2*epsilon3)./(s*s*zeta(2:n+1));

        wT = vT.*Cn(2:n+1) - 2*aas*Bn*(Cn(2:n+1).*chi(2:n+1))*epsilon3/s/s;
        % distinguish between positive and negative velocity terms
        wp = max(wT, 0); wm = min(wT, 0);  % note that these work for vectors vT
        Tz = diff(T);  % T is of length n+2; Tz has length n+1
        Tzm = Tz(1:n); Tzp = Tz(2:n+1);
        Tzup = wp.*Tzm + wm.*Tzp;   % this gives upwind differencing on first deriv terms
        
        Tzz = diff(Tz);  % this is the second derivative wrt chi, since chi spacing is 1
        
        dTdt = -Tzup + epsilon3*Cn(2:n+1).*Cn(2:n+1) .* Tzz/s/s;
        
        % T in magma:  
        
        % use upwinding on the first spatial derivative terms:
        
        vTm = (xi(2:m+1)-1)*dsdt/(1.0-s) - 2*delta5./((1-s)*( s + (1-s)*xi(2:m+1)));
        WTm = vTm.*Cm(2:m+1) - 2*aam*Bm*Cm(2:m+1).*(psi(2:m+1)-m-1)*delta5/(1-s)/(1-s);
        vpm = max(WTm, 0); vmm = min(WTm, 0);  % note that these work for vectors vTm too
        Tmxi = diff(Tm) ;  % Tm is of length n+2; Tmxi has length n+1
        Txim = Tmxi(1:m); Txip = Tmxi(2:m+1);
        Tmup = vpm.*Txim + vmm.*Txip;  % does upwinding on d Tm/ d psi terms

        Tmzz = diff(Tmxi);  % second spatial derivative of Tm
        
        dTmdt = -Tmup + delta5*Cm(2:m+1).*Cm(2:m+1)/((1-s)*(1-s)) .* Tmzz;  %T in magma
        
        % now do the rho equation in the magma. xi is the spatial variable
        % in the magma:
        
        Vrhoi = (xi(2:m+1)-1).*Cm(2:m+1) * dsdt/(1-s);
                 
        vprho = max(Vrhoi, 0); vmrho= min(Vrhoi, 0);
        drho = diff(rho);        
        rhozm = drho(1:m); rhozp= drho(2:m+1);
        rhozup = vprho.*rhozm + vmrho.*rhozp;  % upwinding value of coeff * d rho/ dz
            
        VPrRho=-2*epsilon5./( (1-s)*(s+ (1-s)*xi(2:m+1)));  % the Vp part only
        VPrRho= VPrRho.*Cm(2:m+1)- epsilon5*2*aam*Bm*(Cm(2:m+1).*(psi(2:m+1)-m-1))/(1-s)^2;
        VPrRho = VPrRho.*rho(2:m+1);
        VPpos=max(VPrRho,0); VPneg=min(VPrRho,0);
        dpee = diff(p); % should be m+1 terms in here
        pneg=dpee(1:m); ppos=dpee(2:m+1);
        VPup=VPpos.*pneg + VPneg.*ppos;    % generates the rho*dp/d psi term in rho equation
        
        % a more accurate second difference term uses average rho values:
        
        Pflux = (rho(1:m+1) + rho(2:m+2)).* dpee/2; % the nonlinear flux term
        Peepp = diff(Pflux);
        
        dRhodt = -rhozup -VPup + ...
           epsilon5/( (1-s)*(1-s)) *Cm(2:m+1).*Cm(2:m+1).* Peepp;  % the RHS of density DE
        
        RSide = [dTdt'; dTmdt'; dRhodt'; dsdt];
    end
        

    
    function LookAtFlash
        %% LookAtFlash plots the LHS and RHS of the flash condition
        % this is exploratory
        % I find that for T in the range T0,0.5 (that is, two-phase
        % conditions), there is only one solution with initial conditions
        % in place. This changed however, when temperatures dropped in
        % magma and increased in slurry. The model was flawed though...


        T = linspace(T0/10,0.5,500);

        LHS = LeftSide(T);
        
        RHS = RightSide(T);
        
        
        figure(21)
        clf('reset')
        plot(T,LHS,'-k','LineWidth', 3)
        hold on;
        plot(T,RHS,'-r', 'LineWidth', 3)
        Taxis=min(min(LHS),min(RHS));
        MaxV=max(max(LHS),max(RHS));
        plot([Tnp1, Tnp1],[Taxis, MaxV],'-b')
        plot([Tm2, Tm2],[Taxis, MaxV],'-g')

        ax=gca;
        defFont=24;
        ax.FontSize=defFont; ax.FontName='Times';
        hold off;
        xlabel('T^*');

        figure(22)
        clf('reset')
        plot(T,log10(abs(LHS)),'-k','LineWidth', 3)
        hold on;
        plot(T,log10(abs(RHS)),'-r', 'LineWidth', 3)
        ax=gca;
        defFont=24;
        ax.FontSize=defFont; ax.FontName='Times';
        title('logs taken');
        hold off;
    end
    


    function result =FlashBC(T)
    %% FlashBC computes the balance across the flash
    % this balance will determine the value of temperature there
    %
    % variables passed as globals to LeftSide are p2 and p3, pressures
    % adjacent to flash, and variables passed as globals to RightSide are 
    % temperatures Tnp1 and Tn in slurry, 
    % and temperatures Tm2 and Tm3 in magma, all adjacent to flash
        
        LHS = LeftSide(T);
        
        RHS = RightSide(T);

        result=LHS-RHS;  %fzero will try to find when this value is zero
        
    end

    function LHS = LeftSide(T)
    %% LeftSide computes the left-hand side of the flash condition
        global epsilon4 Haitch St
        
        p=exp(Haitch*(T-T0)./T);
    
    % the following is a first-order difference approximation
    
    %   LHS = St*epsilon4*p.*(p2-p);
 % here is a second-order one-sided difference approximation to the gradient:
 % it requires an extra pressure value adjacent to flash, p3, as well as p2
        LHS = St*epsilon4*p.*(-p3+4*p2-3*p);
 
    end

    function RHS = RightSide(T)
    %% RightSide computes the right-hand side of the flash condition
    %  
        
    % RHS = T.*T/s + T*((s-1)/s *Tnp1 -Tm2);  % this is OK iff m=n
    
% the following formula is first-order accurate, and only requires the adjacent values
% Tm2, Tnp1  for temperature in magma and slurry
%
% RHS = -T.*(Tm2-T+Bn/Bm  * aam/aas *(s-1)/s * ...
%         (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(T-Tnp1));  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)

% the following formula is a second-order one-sided difference. It requires
% values for the location s of the flashing front, and values of 
% Tm2, Tm3, Tnp1, Tn  for temperatures in magma and slurry
% adjacent to flash:

    RHS = -T.*(-Tm3 +4*Tm2-3*T +Bn/Bm  * aam/aas *(s-1)/s * ...
        (1+aas*aas*(n+1)*(n+1))/(1+aam*aam*(m+1)*(m+1)) *(3*T-4*Tnp1+Tn));  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)
               
    end

    function result = FlashBCinit(T)
    %% computes the flash condition for initial setup only, with error fn T similarity solns
    %  the similarity solution is used for T gradient in magma at next node 
    %  and for T gradient in slurry too.
    % tSince is time since emplacement, time since the similarity solution
    % was a step function
        
    global St epsilon4 Haitch
        
    p=exp(Haitch*(T-T0)./T);
    Cm0 = Bm/aam * (1+aam*aam*(m+1)*(m+1));
    LHS =  St*epsilon4* p *(-p3+4*p2-3*p);
    RHS = -2*T/(Cm0*sqrt(pi*tSince)) * (1-s0)*( (1-T)/sqrt(delta5) - (T-T0)/sqrt(epsilon3) );  % allows m not equal to n; and aam not equal to aas (the value of a in the slurry)
    
    
    result = LHS-RHS;           
    end


function  RescaleSurtsey
%% RescaleSurtsey does the rescalings that nondimensionalise the full P,T
% model of steaming surtseyan bombs, and computes parameter values
% 

global Haitch epsilon4 epsilon5 St Tsurface

R2 = 0.1; %bomb size, m
R1 = 0.01; % inclusion size, m
s0=R1/R2;
Tm = 1275.0; % magma temperature, K
Pa = 1.0E05; %scale pressure on atmospheric
Ti = 373; % initial temperature of inclusion, K

rhom = 2750.0;  %density of magma ignoring pores; multipy by porosity to get apparent density
Cpm =  840.0;   %thermal capacity of magma rock, J/kg/K
Cpl = 4200;  %thermal capacity of liquid water, J/kg/K
Cps = 2000; % thermal capacity of water vapour, J/kg/K
Ke=2;    % thermal conductivity, gas filled magma
Kel=3;    % thermal conductivity, liquid filled magma
M=18.0e-03; %molar mass water, kg/mol

%phi = 0.4; %magma porosity
Hsl = 2.3e06; % specific heat of vaporisation for water, J/kg

R=8.314;   % universal gas constant, J/kg/mol

rho_0l = 1000; % reference liquid water density, kg/m^3
muv = 3e-05; % dynamic viscosity Pa.s

Haitch = M*Hsl/(R*Ti);
rhos=Pa*M/(R*Tm);  % called \rho_{0s} in notes
rockheat=(1-phi)*rhom*Cpm;
rscs=phi*rhos*Cps;
rhoc=rockheat + rscs;
rhopCp=rockheat+ phi*rho_0l*Cpl;

t0=phi*rho_0l*Hsl*R1*R1/(3*Ke*(Tm-Ti));
v0s=R2*rho_0l/(t0*rhos);

delta5=Ke*t0/(rhoc*R2*R2);

epsilon3 = Kel*t0/(R2*R2*rhopCp);
St=Hsl*phi*rhos*R2*v0s/(Tm*Ke);
T0=Ti/Tm; % initial temperature in the slurry; boiling point at one atm
Tsurface = T0; % set surface T of magma to boiling point for one atmosphere, and rescale it
epsilon4=k*Pa/(muv*R2*phi*v0s);
epsilon5 = k*Pa*t0/(phi*muv*R2*R2);


% theoretical pmax value:

RampDistance=1E-05/R2; % pore diameter, 10 microns before nondimensionalising

Ap = 3*Ke*muv*sqrt(rhoc*rho_0l)/(pi*sqrt(3)*RampDistance*Pa*Hsl^(3/2)*rhos*rhos);
pmaxcubed = Ap/(k*sqrt(phi))  * Tm*Tm/(sqrt(Tm-Ti)) * R1*R1/R2/R2 * (R2-R1)/R2;
pmaxtheor = pmaxcubed^(1/3);

% theoretical time at which this max pressure is reached:

Bee=sqrt(1.2*s0*(1-s0)/(epsilon4*St*sqrt(pi*delta5)));

Bee2=0.3*epsilon5/(epsilon4*St*sqrt(pi*delta5));

tmaxtheor = (Bee*RampDistance/4/sqrt(delta5)/(2*Bee2))^(4/3);

end

    function y = GetZmag(a) 
        %% the zero of this function determines a in magma
        % and the resulting a gives the correct minimum mesh in magma
       y = 1 - atan(a*m)/atan(a*(m+1)) - ximin; 
    end


%     function y = GetZslurry(en) 
%         %% the zero of this function determines n given aam
%         % and the resulting n gives almost the correct minimum mesh in slurry
%        y = 1 - atan(aam*en)/atan(aam*(en+1)) - zetamin; 
%     end

    function y = GetAAslurry(a) 
        %% the zero of this function determines a in slurry, given n
        % and the resulting a gives the correct minimum mesh in slurry
       y = 1 - atan(a*n)/atan(a*(n+1)) - zetamin; 
    end

    function [val,ist,dir]=MyEvents(t,y) %#ok<DEFNU>
        %% to stop if s gets too small or max p is reached
        val1=y(end)-1.0E-03;  % this is s-smin. dimensionless s
        ist1=1;         %this is terminal; stop if encountered
        dir1=0;         %any direction
        % also stop if max temperature is reached at flash. Use the temperature
        % in the magma as a proxy for this:
        RightSide = SurtseyModel(t,y); % generate the right-hand sides of
                                       % the ODEs
        rhodot=RightSide(n+m+1); % time rate of change of the first rho in magma
                              % next to the flash. This reaches a max at a
                              % time close to and just after when Pflash
                              % does
        Tdot=RightSide(n+1); % time rate of change of T in magma next to flash
        
        % P is rho* T; use product rule to find time rate of change of P in
        % magma:
        T2=y(n+1); Rho2=y(n+m+1); % need actual values too
        val2 = rhodot*T2 + Rho2*Tdot; % rate of change of pressure at first node in magma, next to flash

        ist2=1; % stop if encounter this being zero
        dir2=-1;    % direction is decreasing dT/dt for a maximum of T there
        
        val=[val1; val2];
        ist=[ist1; ist2];
        dir=[dir1; dir2];


    end
end

